const { app, BrowserWindow, ipcMain, globalShortcut, shell } = require('electron');
const path = require('path');
const fs = require('fs');
const { spawn } = require('child_process');

const isDev = !app.isPackaged;
const appDirectory = isDev ? app.getAppPath() : path.dirname(app.getPath('exe'));
const settingsPath = path.join(appDirectory, 'settings.json');
let activeEmuProcess = null;

function logInfo(message) { console.log(`[INFO] ${new Date().toISOString()}: ${message}`); }
function logError(message) { console.error(`[ERROR] ${new Date().toISOString()}: ${message}`); }

function readSettings() {
    let settings = {};
    try {
        if (fs.existsSync(settingsPath)) {
            settings = JSON.parse(fs.readFileSync(settingsPath, 'utf-8'));
        }
    } catch (error) {
        logError(`Error reading settings file: ${error.message}`);

        settings = {};
    }

    if (!settings.netplayUsername) {
        settings.netplayUsername = String(Math.floor(10000 + Math.random() * 90000));
        writeSettings(settings);
        logInfo(`No username found. Generated and saved new username: ${settings.netplayUsername}`);
    }

    return settings;
}

function writeSettings(settings) {
    try {
        fs.writeFileSync(settingsPath, JSON.stringify(settings, null, 2));
    } catch (error) { logError(`Error saving settings file: ${error.message}`); }
}

function synchronizeRegionalFiles() {
    logInfo('Synchronizing regional files...');
    const settings = readSettings();
    const region = (settings.region || 'USA').toLowerCase(); // a 'usa' or 'eur'
    
    const resourcesPath = isDev ? path.join(__dirname, '..', 'resources') : path.join(process.resourcesPath, 'resources');
    const gamesDir = path.join(resourcesPath, 'games', 'SUPER NINTENDO ENTERTAINMENT SYSTEM');

    const filesToSync = ['_games', '_covers'];
    
    filesToSync.forEach(baseName => {
        const sourceFile = `${baseName}_${region}.json`;
        const destFile = `${baseName}.json`;
        const sourcePath = path.join(gamesDir, sourceFile);
        const destPath = path.join(gamesDir, destFile);

        if (fs.existsSync(sourcePath)) {
            try {
                fs.copyFileSync(sourcePath, destPath);
                logInfo(`Successfully copied ${sourceFile} to ${destFile}`);
            } catch (err) {
                logError(`Failed to copy ${sourceFile}: ${err.message}`);
            }
        } else {
            logError(`Source file not found for synchronization: ${sourcePath}`);
        }
    });
}

function isValidServerUrl(url) {
    if (typeof url !== 'string' || !url) return false;
    let cleanedUrl = url.trim().toLowerCase();
    if (cleanedUrl.startsWith('https://')) cleanedUrl = cleanedUrl.substring(8);
    if (cleanedUrl.endsWith('/')) cleanedUrl = cleanedUrl.slice(0, -1);
    return ['erista.me', 'myrient.erista.me'].includes(cleanedUrl);
}

function createWindow() {
    const settings = readSettings();
    const mainWindow = new BrowserWindow({
        width: 1280, 
        height: 850, 
        resizable: true, 
        fullscreenable: true, 
        autoHideMenuBar: true,
        fullscreen: settings.startFullscreen === true,
        webPreferences: {
            preload: path.join(__dirname, 'preload.js'),
            contextIsolation: true,
            nodeIntegration: false,
            webSecurity: !isDev,
        }
    });
    mainWindow.loadFile(path.join(__dirname, 'index.html'));
    
    if (isDev) {
        mainWindow.webContents.openDevTools();
    }
}

let promptWindow = null;
function createUrlPromptWindow() {
    promptWindow = new BrowserWindow({
        width: 600,
        height: 400,
        resizable: false,
        autoHideMenuBar: true,
        parent: null,
        modal: true,
        webPreferences: {
            preload: path.join(__dirname, 'prompt-preload.js'),
            contextIsolation: true,
            nodeIntegration: false,
        }
    });
    promptWindow.loadFile(path.join(__dirname, 'prompt.html'));
    promptWindow.on('closed', () => {
        if (!BrowserWindow.getAllWindows().some(w => w.isVisible())) {
            app.quit();
        }
    });
}

app.whenReady().then(() => {
    synchronizeRegionalFiles(); // Ensure correct files are in place before launching

    const settings = readSettings();
    if (isValidServerUrl(settings.serverUrl)) {
        createWindow();
    } else {
        createUrlPromptWindow();
    }

    globalShortcut.register('F11', () => {
        const win = BrowserWindow.getFocusedWindow();
        if (win) {
            win.setFullScreen(!win.isFullScreen());
        }
    });
});

ipcMain.on('url-submitted', (event, url) => {
    if (isValidServerUrl(url)) {
        const settings = readSettings();
        settings.serverUrl = url;
        writeSettings(settings);
        promptWindow.close();
        createWindow();
    } else {
        event.sender.send('validation-result', { success: false });
    }
});

app.on('will-quit', () => { globalShortcut.unregisterAll(); });
app.on('window-all-closed', () => { if (process.platform !== 'darwin') { app.quit(); } });
app.on('activate', () => { if (BrowserWindow.getAllWindows().length === 0) { createWindow(); } });

ipcMain.on('relaunch-app', () => {
    synchronizeRegionalFiles(); // Sync files with the new region setting before relaunching
    app.relaunch();
    app.quit();
});

ipcMain.on('open-external-link', (event, url) => {
    shell.openExternal(url);
});

ipcMain.handle('get-locale-data', (event, lang) => {
    try {
        const resourcesPath = isDev ? path.join(__dirname, '..', 'resources') : path.join(process.resourcesPath, 'resources');
        const localePath = path.join(resourcesPath, 'locales', `${lang}.json`);
        if (fs.existsSync(localePath)) {
            const data = fs.readFileSync(localePath, 'utf-8');
            return JSON.parse(data);
        }
        logError(`Locale file not found for language: ${lang}`);
        return {};
    } catch (error) {
        logError(`Error reading locale file for ${lang}: ${error.message}`);
        return {};
    }
});

ipcMain.handle('get-initial-data', () => {
    logInfo('Request received for initial app data.');
    try {
        const settings = readSettings();
        const region = settings.region || 'USA';
        const suffix = region === 'Europe' ? '_eur' : '';
        const getRegionalPath = (dir, baseName, ext) => {
            const regionalFile = `${baseName}${suffix}${ext}`;
            if (fs.existsSync(path.join(dir, regionalFile))) {
                return regionalFile;
            }
            return `${baseName}${ext}`;
        };

        const resourcesPath = isDev ? path.join(__dirname, '..', 'resources') : path.join(process.resourcesPath, 'resources');
        const coversDir = path.join(resourcesPath, 'images', 'covers');
        const gamesDir = path.join(resourcesPath, 'games', 'SUPER NINTENDO ENTERTAINMENT SYSTEM');
        const soundsPath = path.join(resourcesPath, 'sounds');
        const customBgmDir = path.join(soundsPath, 'bgm');
        const retroarchPath = path.join(resourcesPath, 'retroarch');
        const toFileUrl = (p) => 'file:///' + p.replace(/\\/g, '/');

        const uiImagesDir = path.join(resourcesPath, 'images', 'ui');
        const uiUrls = {
            navigateSoundUrl: toFileUrl(path.join(soundsPath, 'navigate.wav')),
            confirmSoundUrl: toFileUrl(path.join(soundsPath, 'confirm.wav')),
            cancelSoundUrl: toFileUrl(path.join(soundsPath, 'cancel.wav')),
            bgUrl: toFileUrl(path.join(uiImagesDir, 'bg.png')),
            bg2Url: toFileUrl(path.join(uiImagesDir, getRegionalPath(uiImagesDir, 'bg2', '.png'))),
            sstateUrl: toFileUrl(path.join(uiImagesDir, getRegionalPath(uiImagesDir, 'sstate', '.png'))),
            topGuiUrl: toFileUrl(path.join(uiImagesDir, getRegionalPath(uiImagesDir, 'top_gui', '.png'))),
            bottomGuiUrl: toFileUrl(path.join(uiImagesDir, getRegionalPath(uiImagesDir, 'bottom_gui', '.png'))),
            gamepad1pIconUrl: toFileUrl(path.join(uiImagesDir, getRegionalPath(uiImagesDir, 'gamepad', '.png'))),
            gamepad2pIconUrl: toFileUrl(path.join(uiImagesDir, getRegionalPath(uiImagesDir, 'gamepad-2p', '.png'))),
            menu1Url: toFileUrl(path.join(uiImagesDir, 'menu1.png')),
            menu2Url: toFileUrl(path.join(uiImagesDir, 'menu2.png')),
            menu3Url: toFileUrl(path.join(uiImagesDir, 'menu3.png')),
            menu4Url: toFileUrl(path.join(uiImagesDir, 'menu4.png')),
            menu5Url: toFileUrl(path.join(uiImagesDir, 'menu5.png')),
            gamepadUpUrl: toFileUrl(path.join(uiImagesDir, 'gamepad_up.png')),
            gamepadDownUrl: toFileUrl(path.join(uiImagesDir, 'gamepad_down.png')),
            selectUrl: toFileUrl(path.join(uiImagesDir, 'select.png')),
            startUrl: toFileUrl(path.join(uiImagesDir, 'start.png')),
            marioGifUrl: toFileUrl(path.join(uiImagesDir, 'mario.gif')),
        };

        const appPaths = {
            gamesDir: gamesDir,
            statesDir: path.join(retroarchPath, 'states')
        };
        
        let twoPlayerGames = [];
        const metadataPath = path.join(resourcesPath, 'metadata.json');
        if (fs.existsSync(metadataPath)) {
            const metadataContent = fs.readFileSync(metadataPath, 'utf-8');
            twoPlayerGames = JSON.parse(metadataContent).twoPlayerGames || [];
        } else {
            logError('metadata.json not found!');
        }

        const gamesListPath = path.join(gamesDir, '_games.json'); // Simplified: Always use _games.json
        if (!fs.existsSync(gamesListPath)) {
            logError(`CRITICAL: _games.json not found at ${gamesListPath}`);
            throw new Error('_games.json is missing.');
        }
        const gamesData = JSON.parse(fs.readFileSync(gamesListPath, 'utf-8'));
        
        let coverCache = {};
        const coverCachePath = path.join(gamesDir, '_covers.json'); // Simplified: Always use _covers.json

        if (fs.existsSync(coverCachePath)) {
            logInfo('Loading cover paths from _covers.json cache.');
            coverCache = JSON.parse(fs.readFileSync(coverCachePath, 'utf-8'));
        } else {
            logInfo('Cover cache not found. Generating...');
            const findCoverRelativePath = (gameBaseName) => {
                const simpleBaseName = gameBaseName.match(/^([^([]+)/)?.[0].trim() || gameBaseName;
                const namesToTry = [gameBaseName];
                if (simpleBaseName !== gameBaseName) namesToTry.push(simpleBaseName);
                for (const name of namesToTry) {
                    for (const ext of ['.png', '.jpg']) {
                        const coverPath = path.join(coversDir, name + ext);
                        if (fs.existsSync(coverPath)) {
                            return path.relative(resourcesPath, coverPath).replace(/\\/g, '/');
                        }
                    }
                }
                return null;
            };

            gamesData.forEach(gameEntry => {
                const baseName = path.parse(gameEntry.rom_name).name;
                const relativePath = findCoverRelativePath(baseName);
                if (relativePath) {
                    coverCache[baseName] = relativePath;
                }
            });

            fs.writeFileSync(coverCachePath, JSON.stringify(coverCache, null, 2));
            logInfo('Cover cache generated and saved.');
        }

        const games = gamesData.map(gameEntry => {
            const zipName = gameEntry.rom_name;
            const baseName = path.parse(zipName).name;
            const displayName = baseName.match(/^([^([]+)/)?.[0].trim() || baseName;
            const relativeCoverPath = coverCache[baseName];
            return {
                baseName: displayName,
                zipName: zipName,
                coverUrl: relativeCoverPath ? toFileUrl(path.join(resourcesPath, relativeCoverPath)) : null,
            };
        });
        logInfo(`Found ${games.length} games.`);

        let bgmTracks = [{ name: 'Music Off', path: null }, { name: 'Default', path: path.join(soundsPath, 'bgm.mp3') }];
        if (fs.existsSync(customBgmDir)) {
            const customFiles = fs.readdirSync(customBgmDir)
                .filter(file => file.toLowerCase().endsWith('.mp3'))
                .map(file => ({
                    name: path.parse(file).name,
                    path: path.join(customBgmDir, file)
                }));
            bgmTracks = [...bgmTracks, ...customFiles];
        }

        return { uiUrls, appPaths, games, twoPlayerGames, bgmTracks };
    } catch (error) {
        logError(`Failed to get initial data: ${error.stack}`);
        return null;
    }
});

ipcMain.handle('get-settings', () => readSettings());

ipcMain.on('save-app-settings', (event, settings) => {
    logInfo('Saving application-only settings.');
    writeSettings(settings);
});

ipcMain.on('save-all-settings', (event, settings) => {
    logInfo('Saving all settings and updating RetroArch config.');
    writeSettings(settings);
    writeRetroarchConfig(settings);
});

ipcMain.on('save-settings-on-exit', (event, settings) => {
    logInfo('Saving final settings on application exit.');
    writeSettings(settings);
});

function writeRetroarchConfig(settings) {
    const resourcesPath = isDev ? path.join(__dirname, '..', 'resources') : path.join(process.resourcesPath, 'resources');
    const retroarchCfgPath = path.join(resourcesPath, 'retroarch', 'retroarch.cfg');
    if (!fs.existsSync(retroarchCfgPath)) return;
    let content = fs.readFileSync(retroarchCfgPath, 'utf-8');
    const retroarchKeys = ["savestate_auto_index", "savestate_auto_load", "savestate_max_keep", "aspect_ratio_index", "video_fullscreen", "video_windowed_fullscreen", "input_overlay_enable", "input_overlay_opacity", "rewind_enable", "input_hold_fast_forward_axis", "cheevos_enable", "cheevos_username", "cheevos_password", "savestate_slot", "user_language"];
    for (const key of retroarchKeys) {
        if (settings[key] !== undefined) {
            const value = settings[key];
            const regex = new RegExp(`^(\\s*${key}\\s*=\\s*).*`, 'm');
            const newLine = `${key} = "${value}"`;
            if (regex.test(content)) {
                content = content.replace(regex, newLine);
            } else {
                content += `\n${newLine}`;
            }
        }
    }
    content = content.split('\n').filter(line => !line.trim().startsWith('cheevos_token')).join('\n');
    fs.writeFileSync(retroarchCfgPath, content);
}

function resolveBackendScript(scriptName) {
    const backendPath = isDev ? path.join(__dirname, '..', 'backend') : path.join(process.resourcesPath, 'backend');
    const pyScriptPath = path.join(backendPath, `${scriptName}.py`);
    const exePath = path.join(backendPath, `${scriptName}.exe`);
    if (fs.existsSync(exePath)) {
        return { executable: exePath, scriptArgs: [] };
    } else if (fs.existsSync(pyScriptPath)) {
        return { executable: 'python', scriptArgs: [pyScriptPath] };
    } else {
        logError(`CRITICAL: Backend script not found for '${scriptName}'. Neither .py nor .exe exists in ${backendPath}`);
        const win = BrowserWindow.getFocusedWindow();
        if (win) {
            const errorMsg = `Backend component '${scriptName}' is missing. Please reinstall the application.`;
            win.webContents.send('emu-progress', JSON.stringify({ type: 'error', message: errorMsg }));
        }
        return null;
    }
}

function spawnEmulator(args) {
    const resolvedScript = resolveBackendScript('emu_app');
    if (!resolvedScript) return;
    const command = resolvedScript.executable;
    const finalArgs = [...resolvedScript.scriptArgs, ...args];
    logInfo(`Spawning emulator: ${command} ${finalArgs.join(' ')}`);
    activeEmuProcess = spawn(command, finalArgs);
    activeEmuProcess.stdout.on('data', (data) => {
        const lines = data.toString().split('\n').filter(line => line.trim() !== '');
        lines.forEach(line => {
            BrowserWindow.getAllWindows().forEach(win => win.webContents.send('emu-progress', line));
        });
    });
    activeEmuProcess.stderr.on('data', (data) => { logError(`Backend Error: ${data}`); });
    activeEmuProcess.on('exit', (code) => { 
        logInfo(`Emulator process exited with code: ${code}`);
        BrowserWindow.getAllWindows().forEach(win => win.webContents.send('emu-exit', code)); 
        activeEmuProcess = null;
    });
}

ipcMain.on('cancel-emu-process', () => {
    if (activeEmuProcess) {
        logInfo('Received request to cancel emulator process.');
        activeEmuProcess.kill();
        activeEmuProcess = null;
    }
});

function handleGameLaunch(options) {
    const { gamePath, stateSlot, targetSlot, mode } = options;
    if (!gamePath) return;
    const currentSettings = readSettings();
    const configForRetroarch = { ...currentSettings };
    
    if (targetSlot !== undefined && targetSlot !== null) {
        configForRetroarch.savestate_slot = targetSlot.toString();
    } else {
        delete configForRetroarch.savestate_slot;
    }

    writeRetroarchConfig(configForRetroarch);
    const resourcesPath = isDev ? path.join(__dirname, '..', 'resources') : path.join(process.resourcesPath, 'resources');
    const statesDir = path.join(resourcesPath, 'retroarch', 'states');
    const gameBaseName = path.parse(gamePath).name;
    const autoSavePath = path.join(statesDir, `${gameBaseName}.state.auto`);
    try {
        if (fs.existsSync(autoSavePath)) fs.unlinkSync(autoSavePath);
        if (stateSlot !== undefined && stateSlot !== null) {
            const slotSuffix = stateSlot > 0 ? `${stateSlot}` : '';
            const sourceSavePath = path.join(statesDir, `${gameBaseName}.state${slotSuffix}`);
            if (fs.existsSync(sourceSavePath)) {
                fs.copyFileSync(sourceSavePath, autoSavePath);
            }
        }
    } catch (err) {
        logError(`Error managing save state files: ${err.message}`);
    }
    const spawnArgs = [gamePath];
    if (mode === 'play-netplay-host') {
        spawnArgs.unshift('--mode', 'play-netplay-host', '--username', currentSettings.netplayUsername || 'Player1');
    } else if (mode === 'play-netplay-join') {
        spawnArgs.unshift('--mode', 'play-netplay-join', '--ip', options.ip, '--port', options.port);
        if (options.mitm_session) {
            spawnArgs.push('--mitm-ip', options.mitm_ip);
            spawnArgs.push('--mitm-session', options.mitm_session);
        }
    } else {
        spawnArgs.unshift('--mode', 'play');
    }
    spawnEmulator(spawnArgs);
}

ipcMain.on('launch-game', (event, options) => handleGameLaunch({ ...options, mode: 'play' }));
ipcMain.on('save-game', (event, gamePath) => { if (!gamePath) return; spawnEmulator(['--mode', 'save-app', gamePath]); });
ipcMain.on('download-collection', () => { spawnEmulator(['--mode', 'download-all']); });
ipcMain.on('netplay-host', (event, options) => handleGameLaunch({ ...options, mode: 'play-netplay-host' }));
ipcMain.on('join-netplay-game', (event, options) => handleGameLaunch({ ...options, mode: 'play-netplay-join' }));

ipcMain.handle('get-lobby-list', () => { 
    return new Promise((resolve, reject) => {
        const resolvedScript = resolveBackendScript('lobby');
        if (!resolvedScript) {
            reject('Lobby script not found.');
            return;
        }
        const child = spawn(resolvedScript.executable, resolvedScript.scriptArgs);
        let stdoutData = ''; 
        child.stdout.on('data', (data) => { stdoutData += data.toString(); });
        child.stderr.on('data', (data) => logError(`Lobby script stderr: ${data}`));
        child.on('exit', (code) => { 
            if (code === 0) { 
                try { resolve(JSON.parse(stdoutData)); } catch (e) { reject('Failed to parse lobby JSON'); } 
            } else { 
                reject(`Lobby script exited with code: ${code}`); 
            } 
        }); 
        child.on('error', (err) => { reject(`Failed to start lobby script: ${err.message}`); });
    }); 
});

ipcMain.handle('manage-save-state', (event, { action, gameBaseName, slot }) => {
    const resourcesPath = isDev ? path.join(__dirname, '..', 'resources') : path.join(process.resourcesPath, 'resources');
    const statesDir = path.join(resourcesPath, 'retroarch', 'states');
    
    if (action === 'scan') {
        try {
            const files = fs.readdirSync(statesDir);
            const discoveredStates = [];
            for (const file of files) {
                const fullPath = path.join(statesDir, file);
                let stateInfo = null;
                if (file.startsWith(gameBaseName) && file.endsWith('.state') && fs.existsSync(`${fullPath}.png`)) {
                    const stats = fs.statSync(fullPath);
                    stateInfo = { fileSlot: 0, mtime: stats.mtime.getTime() };
                } else if (file.startsWith(`${gameBaseName}.state`) && !file.endsWith('.png') && !file.endsWith('.auto')) {
                    const match = file.match(/\.state(\d+)$/);
                    if (match && fs.existsSync(`${fullPath}.png`)) {
                        const slotNumber = parseInt(match[1], 10);
                        const stats = fs.statSync(fullPath);
                        stateInfo = { fileSlot: slotNumber, mtime: stats.mtime.getTime() };
                    }
                }
                if (stateInfo) discoveredStates.push(stateInfo);
            }
            discoveredStates.sort((a, b) => b.mtime - a.mtime);
            const results = Array(4).fill(null);
            for (let i = 0; i < 4 && i < discoveredStates.length; i++) {
                results[i] = discoveredStates[i];
            }
            return results;
        } catch (err) {
            logError(`Error scanning for save states: ${err.message}`);
            return Array(4).fill(null);
        }
    }
    
    if (action === 'delete') {
        try {
            const slotSuffix = slot > 0 ? `${slot}` : '';
            const statePath = path.join(statesDir, `${gameBaseName}.state${slotSuffix}`);
            const pngPath = `${statePath}.png`;
            if (fs.existsSync(statePath)) fs.unlinkSync(statePath);
            if (fs.existsSync(pngPath)) fs.unlinkSync(pngPath);
            return { success: true };
        } catch (err) {
            logError(`Error deleting save state for slot ${slot}: ${err.message}`);
            return { success: false, error: err.message };
        }
    }

    return { success: false, error: 'Unknown action' };
});

function getRetroArchConfigPath() {
    const resourcesPath = isDev ? path.join(__dirname, '..', 'resources') : path.join(process.resourcesPath, 'resources');
    return path.join(resourcesPath, 'retroarch', 'retroarch.cfg');
}

ipcMain.handle('get-relay-config', () => {
    const cfgPath = getRetroArchConfigPath();
    const defaultConfig = { server: "", enabled: "false" };
    if (!fs.existsSync(cfgPath)) return defaultConfig;

    try {
        const content = fs.readFileSync(cfgPath, 'utf-8');
        const lines = content.split('\n');
        let config = { ...defaultConfig };
        
        const enabledLine = lines.find(line => line.trim().startsWith('netplay_use_relay_server'));
        if (enabledLine) {
            const match = enabledLine.match(/"(true|false)"/);
            if (match) config.enabled = match[1];
        }

        const serverLine = lines.find(line => line.trim().startsWith('netplay_mitm_server'));
        if (serverLine) {
            const match = serverLine.match(/"([^"]+)"/);
            if (match) config.server = match[1];
        }
        return config;
    } catch (error) {
        logError(`Error reading retroarch.cfg: ${error.message}`);
        return defaultConfig;
    }
});

ipcMain.on('set-relay-config', (event, { server, enabled }) => {
    const cfgPath = getRetroArchConfigPath();
    if (!fs.existsSync(cfgPath)) {
        logError("retroarch.cfg not found, cannot set relay config.");
        return;
    }
    
    try {
        const content = fs.readFileSync(cfgPath, 'utf-8');
        let lines = content.split('\n');
        let enabledFound = false;
        let serverFound = false;

        lines = lines.map(line => {
            const trimmedLine = line.trim();
            if (trimmedLine.startsWith('netplay_use_relay_server')) {
                enabledFound = true;
                return `netplay_use_relay_server = "${enabled}"`;
            }
            if (trimmedLine.startsWith('netplay_mitm_server')) {
                serverFound = true;
                return `netplay_mitm_server = "${server}"`;
            }
            return line;
        });

        if (!enabledFound) lines.push(`netplay_use_relay_server = "${enabled}"`);
        if (!serverFound) lines.push(`netplay_mitm_server = "${server}"`);

        fs.writeFileSync(cfgPath, lines.join('\n').replace(/\r/g, ''));
    } catch (error) {
        logError(`Error writing to retroarch.cfg: ${error.message}`);
    }
});

ipcMain.on('set-fullscreen', (event, isFullscreen) => { const win = BrowserWindow.getFocusedWindow(); if (win) { win.setFullScreen(isFullscreen); } });
ipcMain.on('exit-app', () => { app.quit(); });
ipcMain.on('suspend-pc', () => { if (process.platform === 'win32') { spawn('powershell.exe -Command "Add-Type -Assembly System.Windows.Forms; [System.Windows.Forms.Application]::SetSuspendState(\'Suspend\', $false, $true)"', { shell: true }); } });
ipcMain.on('power-off-pc', () => { if (process.platform === 'win32') { spawn('shutdown /s /t 0', { shell: true }); } });